import type { Express } from "express";
import { createServer, type Server } from "http";
import { storage } from "./storage";
import { insertContactRequestSchema } from "@shared/schema";
import { z } from "zod";
import { sendContactNotification } from "./email";
import { createExcelAndSend } from "./excel-service";

export async function registerRoutes(app: Express): Promise<Server> {
  // Contact form submission endpoint
  app.post("/api/contact", async (req, res) => {
    try {
      const validatedData = insertContactRequestSchema.parse(req.body);
      const contactRequest = await storage.createContactRequest(validatedData);
      
      // Enviar email de notificação
      try {
        await sendContactNotification({
          name: validatedData.name,
          email: validatedData.email,
          phone: validatedData.phone,
          projectType: validatedData.projectType,
          message: validatedData.message || "",
          source: (req.body as any).source || "website",
        });
      } catch (emailError) {
        console.error("Erro ao enviar email (mas a solicitação foi salva):", emailError);
        // Não falhamos a requisição se o email falhar, apenas logamos o erro
      }
      
      res.json({ 
        success: true, 
        message: "Solicitação enviada com sucesso! Entraremos em contato em breve.",
        id: contactRequest.id 
      });
    } catch (error) {
      if (error instanceof z.ZodError) {
        res.status(400).json({ 
          success: false, 
          message: "Dados inválidos",
          errors: error.errors 
        });
      } else {
        console.error("Error creating contact request:", error);
        res.status(500).json({ 
          success: false, 
          message: "Erro interno do servidor" 
        });
      }
    }
  });

  // Get all contact requests (for admin purposes)
  app.get("/api/contact-requests", async (req, res) => {
    try {
      const requests = await storage.getAllContactRequests();
      res.json(requests);
    } catch (error) {
      console.error("Error fetching contact requests:", error);
      res.status(500).json({ 
        success: false, 
        message: "Erro ao buscar solicitações" 
      });
    }
  });

  // Endpoint to send Excel file via email
  app.post("/api/send-excel", async (req, res) => {
    try {
      const { name, email, phone, address } = req.body;

      // Validate required fields
      if (!name || !email || !phone || !address) {
        res.status(400).json({
          success: false,
          message: "Todos os campos são obrigatórios",
        });
        return;
      }

      // Validate email format
      const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
      if (!emailRegex.test(email)) {
        res.status(400).json({
          success: false,
          message: "E-mail inválido",
        });
        return;
      }

      // Create Excel and send email
      await createExcelAndSend({ name, email, phone, address });

      res.json({
        success: true,
        message: "Dados enviados com sucesso!",
      });
    } catch (error: any) {
      console.error("Error in send-excel:", error);
      res.status(500).json({
        success: false,
        message: "Erro ao processar e enviar dados. Por favor, tente novamente.",
      });
    }
  });

  // Proxy endpoint for PHP email sending
  // This will forward requests to send-email.php if PHP is available
  // Otherwise, it falls back to Node.js email service
  app.post("/send-email.php", async (req, res) => {
    try {
      // Try to use Node.js email service first
      const validatedData = {
        name: req.body.name,
        email: req.body.email,
        phone: req.body.phone,
        projectType: req.body.projectType,
        message: req.body.message || "",
        source: req.body.source || "php-form",
      };

      // Validate required fields
      if (!validatedData.name || !validatedData.email || !validatedData.phone || !validatedData.projectType) {
        res.status(400).json({ 
          success: false, 
          message: "Campos obrigatórios faltando" 
        });
        return;
      }

      // Use Node.js email service
      try {
        await sendContactNotification(validatedData);
        res.json({ 
          success: true, 
          message: "Solicitação enviada com sucesso! Entraremos em contato em breve." 
        });
      } catch (emailError) {
        console.error("Erro ao enviar email:", emailError);
        res.status(500).json({ 
          success: false, 
          message: "Erro ao enviar email. Por favor, tente novamente." 
        });
      }
    } catch (error: any) {
      console.error("Error in send-email.php proxy:", error);
      res.status(500).json({ 
        success: false, 
        message: "Erro ao processar solicitação" 
      });
    }
  });

  const httpServer = createServer(app);
  return httpServer;
}
