import * as XLSX from "xlsx";
import nodemailer from "nodemailer";

interface FormData {
  name: string;
  email: string;
  phone: string;
  address: string;
}

export async function createExcelAndSend(formData: FormData): Promise<void> {
  // Create workbook
  const workbook = XLSX.utils.book_new();

  // Create worksheet data
  const worksheetData = [
    ["Nome", "E-mail", "Telefone", "Endereço"],
    [formData.name, formData.email, formData.phone, formData.address],
  ];

  // Create worksheet
  const worksheet = XLSX.utils.aoa_to_sheet(worksheetData);

  // Set column widths
  worksheet["!cols"] = [
    { wch: 20 }, // Nome
    { wch: 30 }, // E-mail
    { wch: 15 }, // Telefone
    { wch: 40 }, // Endereço
  ];

  // Add worksheet to workbook
  XLSX.utils.book_append_sheet(workbook, worksheet, "Dados do Cliente");

  // Generate Excel buffer
  const excelBuffer = XLSX.write(workbook, {
    type: "buffer",
    bookType: "xlsx",
  });

  // Create transporter
  const transporter = nodemailer.createTransport({
    service: "gmail",
    auth: {
      user: process.env.EMAIL_USER || "seu-email@gmail.com",
      pass: process.env.EMAIL_PASS || "sua-app-password",
    },
  });

  // Email configuration
  const recipientEmail = "vgsalatiel@gmail.com";
  const subject = `Nova Solicitação - ${formData.name}`;

  const htmlContent = `
    <!DOCTYPE html>
    <html>
    <head>
      <meta charset="utf-8">
      <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .header { background: linear-gradient(135deg, #3b82f6 0%, #8b5cf6 100%); color: white; padding: 20px; border-radius: 8px 8px 0 0; }
        .content { background: #f9fafb; padding: 20px; border: 1px solid #e5e7eb; }
        .field { margin-bottom: 15px; }
        .label { font-weight: bold; color: #1f2937; }
        .value { color: #4b5563; margin-top: 5px; }
        .footer { background: #1f2937; color: white; padding: 15px; text-align: center; border-radius: 0 0 8px 8px; font-size: 12px; }
      </style>
    </head>
    <body>
      <div class="container">
        <div class="header">
          <h2>Nova Solicitação via Chat Bot</h2>
        </div>
        <div class="content">
          <div class="field">
            <div class="label">Nome:</div>
            <div class="value">${formData.name}</div>
          </div>
          <div class="field">
            <div class="label">E-mail:</div>
            <div class="value">${formData.email}</div>
          </div>
          <div class="field">
            <div class="label">Telefone:</div>
            <div class="value">${formData.phone}</div>
          </div>
          <div class="field">
            <div class="label">Endereço:</div>
            <div class="value">${formData.address}</div>
          </div>
          <div class="field">
            <div class="label">Arquivo Excel:</div>
            <div class="value">Ver anexo</div>
          </div>
        </div>
        <div class="footer">
          <p>Este email foi enviado automaticamente pelo sistema KRTI Network Solutions</p>
        </div>
      </div>
    </body>
    </html>
  `;

  // Send email with Excel attachment
  await transporter.sendMail({
    from: process.env.EMAIL_FROM || process.env.EMAIL_USER || "noreply@krti.com.br",
    to: recipientEmail,
    subject,
    html: htmlContent,
    attachments: [
      {
        filename: `dados-cliente-${Date.now()}.xlsx`,
        content: excelBuffer,
      },
    ],
  });

  console.log("Excel enviado com sucesso para:", recipientEmail);
}

