import nodemailer from "nodemailer";

// Configuração do transporter de email
// Para testes, você pode usar serviços como Gmail, Outlook, ou serviços de email transacional
const createTransporter = () => {
  // Configuração para Gmail (exemplo)
  // Você precisará configurar uma "App Password" no Gmail
  return nodemailer.createTransport({
    service: "gmail",
    auth: {
      user: process.env.EMAIL_USER || "seu-email@gmail.com",
      pass: process.env.EMAIL_PASS || "sua-app-password",
    },
  });

  // Alternativa: SMTP genérico
  // return nodemailer.createTransport({
  //   host: process.env.SMTP_HOST || "smtp.gmail.com",
  //   port: parseInt(process.env.SMTP_PORT || "587"),
  //   secure: false, // true para 465, false para outras portas
  //   auth: {
  //     user: process.env.EMAIL_USER,
  //     pass: process.env.EMAIL_PASS,
  //   },
  // });
};

interface EmailData {
  to: string;
  subject: string;
  html: string;
  text?: string;
}

export async function sendEmail(data: EmailData): Promise<void> {
  try {
    const transporter = createTransporter();
    
    const mailOptions = {
      from: process.env.EMAIL_FROM || process.env.EMAIL_USER || "noreply@krti.com.br",
      to: data.to,
      subject: data.subject,
      html: data.html,
      text: data.text || data.html.replace(/<[^>]*>/g, ""), // Versão texto simples
    };

    const info = await transporter.sendMail(mailOptions);
    console.log("Email enviado com sucesso:", info.messageId);
  } catch (error) {
    console.error("Erro ao enviar email:", error);
    throw error;
  }
}

export async function sendContactNotification(
  contactData: {
    name: string;
    email: string;
    phone: string;
    projectType: string;
    message?: string;
    source?: string;
  }
): Promise<void> {
  // Email padrão para testes - sempre enviar para vgsalatiel@gmail.com
  const recipientEmail = "vgsalatiel@gmail.com";
  
  const subject = `Nova Solicitação de Orçamento - ${contactData.projectType}`;
  
  const html = `
    <!DOCTYPE html>
    <html>
    <head>
      <meta charset="utf-8">
      <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .header { background: linear-gradient(135deg, #3b82f6 0%, #8b5cf6 100%); color: white; padding: 20px; border-radius: 8px 8px 0 0; }
        .content { background: #f9fafb; padding: 20px; border: 1px solid #e5e7eb; }
        .field { margin-bottom: 15px; }
        .label { font-weight: bold; color: #1f2937; }
        .value { color: #4b5563; margin-top: 5px; }
        .footer { background: #1f2937; color: white; padding: 15px; text-align: center; border-radius: 0 0 8px 8px; font-size: 12px; }
      </style>
    </head>
    <body>
      <div class="container">
        <div class="header">
          <h2>Nova Solicitação de Orçamento</h2>
        </div>
        <div class="content">
          <div class="field">
            <div class="label">Nome:</div>
            <div class="value">${contactData.name}</div>
          </div>
          <div class="field">
            <div class="label">E-mail:</div>
            <div class="value">${contactData.email}</div>
          </div>
          <div class="field">
            <div class="label">Telefone:</div>
            <div class="value">${contactData.phone}</div>
          </div>
          <div class="field">
            <div class="label">Tipo de Projeto:</div>
            <div class="value">${contactData.projectType}</div>
          </div>
          ${contactData.message ? `
          <div class="field">
            <div class="label">Mensagem:</div>
            <div class="value">${contactData.message}</div>
          </div>
          ` : ""}
          ${contactData.source ? `
          <div class="field">
            <div class="label">Origem:</div>
            <div class="value">${contactData.source}</div>
          </div>
          ` : ""}
        </div>
        <div class="footer">
          <p>Este email foi enviado automaticamente pelo sistema KRTI Network Solutions</p>
        </div>
      </div>
    </body>
    </html>
  `;

  await sendEmail({
    to: recipientEmail,
    subject,
    html,
  });
}

