import { useState } from "react";
import Navigation from "@/components/navigation";
import Footer from "@/components/footer";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { useToast } from "@/hooks/use-toast";
import { useMutation } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Mail, Send, CheckCircle } from "lucide-react";
import AnimateOnScroll from "@/components/animate-on-scroll";

export default function TestEmail() {
  const { toast } = useToast();
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    projectType: "automacao-completa",
    message: "",
  });

  const sendEmailMutation = useMutation({
    mutationFn: async (data: typeof formData) => {
      const response = await fetch("/send-email.php", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          ...data,
          source: "test-email-form",
        }),
      });
      return response.json();
    },
    onSuccess: (data) => {
      toast({
        title: "Email enviado com sucesso!",
        description: "O email foi enviado para vgsalatiel@gmail.com",
        variant: "default",
      });
      setFormData({
        name: "",
        email: "",
        phone: "",
        projectType: "automacao-completa",
        message: "",
      });
    },
    onError: (error) => {
      toast({
        title: "Erro ao enviar email",
        description: "Ocorreu um erro ao enviar o email. Verifique o console para mais detalhes.",
        variant: "destructive",
      });
      console.error("Email error:", error);
    },
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!formData.name || !formData.email || !formData.phone) {
      toast({
        title: "Campos obrigatórios",
        description: "Por favor, preencha todos os campos obrigatórios.",
        variant: "destructive",
      });
      return;
    }
    sendEmailMutation.mutate(formData);
  };

  const handleInputChange = (field: string, value: string) => {
    setFormData((prev) => ({ ...prev, [field]: value }));
  };

  return (
    <div className="min-h-screen bg-background">
      <Navigation />
      <main>
        {/* Hero Section */}
        <section className="relative py-20 lg:py-32 bg-gradient-to-br from-slate-900 via-blue-900 to-slate-900 overflow-hidden">
          <div className="absolute inset-0 tech-gradient"></div>
          <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
            <AnimateOnScroll direction="up" delay={0.1}>
              <h1 className="text-4xl lg:text-6xl font-bold text-white mb-6">
                <span className="tech-text-gradient">Formulário de Teste de Email</span>
              </h1>
              <p className="text-xl text-gray-300 max-w-3xl mx-auto">
                Use este formulário para testar o envio de emails. 
                Todos os emails serão enviados para <strong className="text-primary">vgsalatiel@gmail.com</strong>
              </p>
            </AnimateOnScroll>
          </div>
        </section>

        {/* Form Section */}
        <section className="py-16 lg:py-24 bg-neutral">
          <div className="max-w-2xl mx-auto px-4 sm:px-6 lg:px-8">
            <AnimateOnScroll direction="up" delay={0.2}>
              <Card className="shadow-lg glass-tech tech-border-glow">
                <CardHeader>
                  <CardTitle className="text-2xl font-semibold text-foreground flex items-center gap-2">
                    <Mail className="w-6 h-6 text-primary" />
                    Teste de Envio de Email
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <form onSubmit={handleSubmit} className="space-y-6">
                    <div className="grid md:grid-cols-2 gap-6">
                      <div>
                        <label className="block text-sm font-medium text-foreground mb-2">
                          Nome *
                        </label>
                        <Input
                          type="text"
                          placeholder="Seu nome completo"
                          value={formData.name}
                          onChange={(e) => handleInputChange("name", e.target.value)}
                          required
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-foreground mb-2">
                          Telefone *
                        </label>
                        <Input
                          type="tel"
                          placeholder="(11) 99999-9999"
                          value={formData.phone}
                          onChange={(e) => handleInputChange("phone", e.target.value)}
                          required
                        />
                      </div>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-foreground mb-2">
                        E-mail *
                      </label>
                      <Input
                        type="email"
                        placeholder="seu@email.com"
                        value={formData.email}
                        onChange={(e) => handleInputChange("email", e.target.value)}
                        required
                      />
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-foreground mb-2">
                        Tipo de Projeto *
                      </label>
                      <select
                        value={formData.projectType}
                        onChange={(e) => handleInputChange("projectType", e.target.value)}
                        className="flex h-10 w-full rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2"
                        required
                      >
                        <option value="automacao-completa">Automação Completa</option>
                        <option value="iluminacao-inteligente">Iluminação Inteligente</option>
                        <option value="seguranca">Segurança</option>
                        <option value="audio-video">Audio & Vídeo</option>
                        <option value="climatizacao">Climatização</option>
                        <option value="outro">Outro</option>
                      </select>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-foreground mb-2">
                        Mensagem
                      </label>
                      <Textarea
                        rows={4}
                        placeholder="Descreva seu projeto e necessidades..."
                        value={formData.message}
                        onChange={(e) => handleInputChange("message", e.target.value)}
                      />
                    </div>

                    <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                      <div className="flex items-start gap-3">
                        <CheckCircle className="w-5 h-5 text-blue-600 dark:text-blue-400 mt-0.5 flex-shrink-0" />
                        <div>
                          <p className="text-sm font-medium text-blue-900 dark:text-blue-100">
                            Email será enviado para:
                          </p>
                          <p className="text-sm text-blue-700 dark:text-blue-300 mt-1">
                            vgsalatiel@gmail.com
                          </p>
                        </div>
                      </div>
                    </div>

                    <Button
                      type="submit"
                      className="w-full bg-primary text-white hover:bg-blue-700 glow-on-hover"
                      disabled={sendEmailMutation.isPending}
                    >
                      {sendEmailMutation.isPending ? (
                        <>
                          <span className="animate-spin mr-2">⏳</span>
                          Enviando...
                        </>
                      ) : (
                        <>
                          <Send className="w-4 h-4 mr-2" />
                          Enviar Email de Teste
                        </>
                      )}
                    </Button>
                  </form>
                </CardContent>
              </Card>
            </AnimateOnScroll>
          </div>
        </section>
      </main>
      <Footer />
    </div>
  );
}

