import { useEffect } from "react";
import Navigation from "@/components/navigation";
import Footer from "@/components/footer";
import { Card, CardContent } from "@/components/ui/card";
import { Mail, Instagram, Clock, MapPin, Phone, MessageCircle } from "lucide-react";
import AnimateOnScroll from "@/components/animate-on-scroll";
import QuoteForm from "@/components/quote-form";

export default function Contact() {
  useEffect(() => {
    document.title = "Contato | KRTI Network Solutions - Solicite seu Orçamento";
    
    const metaDescription = document.querySelector('meta[name="description"]');
    if (metaDescription) {
      metaDescription.setAttribute('content', 'Entre em contato com a KRTI Network Solutions. Solicite um orçamento gratuito para automação residencial. Consultoria especializada e suporte técnico 24/7.');
    }
    
    const metaKeywords = document.querySelector('meta[name="keywords"]');
    if (metaKeywords) {
      metaKeywords.setAttribute('content', 'contato KRTI, orçamento automação residencial, consultoria smart home, suporte técnico automação, WhatsApp automação');
    }
    
    const ogTitle = document.querySelector('meta[property="og:title"]');
    if (ogTitle) {
      ogTitle.setAttribute('content', 'Contato | KRTI Network Solutions');
    }
    
    const ogDescription = document.querySelector('meta[property="og:description"]');
    if (ogDescription) {
      ogDescription.setAttribute('content', 'Solicite um orçamento gratuito para automação residencial. Consultoria especializada e suporte técnico.');
    }
  }, []);

  const whatsappNumber = "5511999999999"; // Substitua pelo número real
  const whatsappMessage = encodeURIComponent(
    `Olá! Gostaria de solicitar um orçamento para automação residencial.`
  );
  const whatsappUrl = `https://wa.me/${whatsappNumber}?text=${whatsappMessage}`;

  return (
    <div className="min-h-screen bg-background">
      <Navigation />
      <main>
        {/* Hero Section */}
        <section className="relative py-20 lg:py-32 bg-gradient-to-br from-slate-900 via-blue-900 to-slate-900 overflow-hidden">
          <div className="absolute inset-0 tech-gradient"></div>
          <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
            <AnimateOnScroll direction="up" delay={0.1}>
              <h1 className="text-4xl lg:text-6xl font-bold text-white mb-6">
                <span className="tech-text-gradient">Entre em Contato</span>
              </h1>
              <p className="text-xl text-gray-300 max-w-3xl mx-auto">
                Pronto para transformar sua casa? Entre em contato conosco para uma consultoria gratuita 
                e descubra como podemos criar o ambiente inteligente dos seus sonhos.
              </p>
            </AnimateOnScroll>
          </div>
        </section>

        {/* Formulário e Informações */}
        <section className="py-16 lg:py-24 bg-neutral">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="lg:grid lg:grid-cols-2 lg:gap-12">
              <AnimateOnScroll direction="right" delay={0.2}>
                <QuoteForm 
                  title="Solicite um Orçamento Gratuito"
                  source="contact-page"
                />
              </AnimateOnScroll>
              
              <div className="mt-12 lg:mt-0 space-y-8">
                <AnimateOnScroll direction="left" delay={0.3}>
                  <Card className="shadow-lg glass-tech tech-border-glow">
                    <CardContent className="p-8">
                      <h3 className="text-2xl font-semibold text-foreground mb-6">Informações de Contato</h3>
                      
                      <div className="space-y-6">
                        <div className="flex items-start">
                          <div className="w-12 h-12 bg-primary/10 rounded-lg flex items-center justify-center mr-4 flex-shrink-0">
                            <Mail className="w-6 h-6 text-primary tech-icon-glow" />
                          </div>
                          <div>
                            <h4 className="font-semibold text-foreground mb-1">E-mail</h4>
                            <a href="mailto:suporte@krti.com.br" className="text-primary hover:text-blue-400">
                              suporte@krti.com.br
                            </a>
                          </div>
                        </div>
                        
                        <div className="flex items-start">
                          <div className="w-12 h-12 bg-primary/10 rounded-lg flex items-center justify-center mr-4 flex-shrink-0">
                            <Phone className="w-6 h-6 text-primary tech-icon-glow" />
                          </div>
                          <div>
                            <h4 className="font-semibold text-foreground mb-1">Telefone</h4>
                            <a href="tel:+5511999999999" className="text-primary hover:text-blue-400">
                              (11) 99999-9999
                            </a>
                          </div>
                        </div>
                        
                        <div className="flex items-start">
                          <div className="w-12 h-12 bg-primary/10 rounded-lg flex items-center justify-center mr-4 flex-shrink-0">
                            <MessageCircle className="w-6 h-6 text-primary tech-icon-glow" />
                          </div>
                          <div>
                            <h4 className="font-semibold text-foreground mb-1">WhatsApp</h4>
                            <a 
                              href={whatsappUrl}
                              target="_blank"
                              rel="noopener noreferrer"
                              className="text-primary hover:text-blue-400"
                            >
                              Falar no WhatsApp
                            </a>
                          </div>
                        </div>
                        
                        <div className="flex items-start">
                          <div className="w-12 h-12 bg-primary/10 rounded-lg flex items-center justify-center mr-4 flex-shrink-0">
                            <Instagram className="w-6 h-6 text-primary tech-icon-glow" />
                          </div>
                          <div>
                            <h4 className="font-semibold text-foreground mb-1">Instagram</h4>
                            <a 
                              href="https://www.instagram.com/krtinetworksolutions/" 
                              target="_blank"
                              rel="noopener noreferrer"
                              className="text-primary hover:text-blue-400"
                            >
                              @krtinetworksolutions
                            </a>
                          </div>
                        </div>
                        
                        <div className="flex items-start">
                          <div className="w-12 h-12 bg-primary/10 rounded-lg flex items-center justify-center mr-4 flex-shrink-0">
                            <Clock className="w-6 h-6 text-primary tech-icon-glow" />
                          </div>
                          <div>
                            <h4 className="font-semibold text-foreground mb-1">Horário de Atendimento</h4>
                            <p className="text-foreground/80">
                              Segunda a Sexta: 8h às 18h<br />
                              Sábado: 8h às 12h
                            </p>
                          </div>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                </AnimateOnScroll>
                
                <AnimateOnScroll direction="left" delay={0.4}>
                  <div className="bg-gradient-to-br from-primary via-blue-600 to-purple-600 rounded-2xl p-8 text-white tech-border-glow glow-primary">
                    <h3 className="text-2xl font-semibold mb-4">Consultoria Gratuita</h3>
                    <p className="mb-6 text-blue-100">
                      Agende uma visita gratuita para avaliarmos suas necessidades e 
                      apresentarmos a melhor solução em automação residencial.
                    </p>
                    <a 
                      href={whatsappUrl}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="inline-block bg-white text-primary hover:bg-gray-100 px-6 py-3 rounded-lg font-medium transition-colors glow-on-hover"
                    >
                      Agendar Consultoria
                    </a>
                  </div>
                </AnimateOnScroll>
              </div>
            </div>
          </div>
        </section>

        {/* Área de Cobertura */}
        <section className="py-16 lg:py-24 bg-background">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <AnimateOnScroll direction="up" delay={0.1}>
              <div className="text-center mb-12">
                <h2 className="text-3xl lg:text-4xl font-bold mb-4">
                  <span className="tech-text-gradient">Nossa Área de Atuação</span>
                </h2>
                <p className="text-xl text-foreground/90 max-w-3xl mx-auto">
                  Atendemos em toda a região metropolitana e interior de São Paulo
                </p>
              </div>
            </AnimateOnScroll>

            <div className="grid md:grid-cols-3 gap-8">
              {[
                { title: "Grande São Paulo", desc: "Atendimento completo em toda a região metropolitana" },
                { title: "Interior de SP", desc: "Projetos em cidades do interior paulista" },
                { title: "Outras Regiões", desc: "Consultas sobre atendimento em outras localidades" },
              ].map((item, index) => (
                <AnimateOnScroll key={index} direction="up" delay={0.2 + index * 0.1}>
                  <div className="glass-tech rounded-xl p-6 tech-border-glow text-center">
                    <MapPin className="w-12 h-12 text-primary mx-auto mb-4 tech-icon-glow" />
                    <h3 className="text-lg font-semibold text-foreground mb-2">{item.title}</h3>
                    <p className="text-foreground/80">{item.desc}</p>
                  </div>
                </AnimateOnScroll>
              ))}
            </div>
          </div>
        </section>
      </main>
      <Footer />
    </div>
  );
}


