import { useState, useRef, useEffect } from "react";
import { MessageCircle, X, Send, Bot, User } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { useToast } from "@/hooks/use-toast";
import { motion, AnimatePresence } from "framer-motion";

interface ChatMessage {
  type: "bot" | "user";
  text: string;
}

interface FormData {
  name: string;
  email: string;
  phone: string;
  address: string;
}

export default function ChatBot() {
  const [isOpen, setIsOpen] = useState(false); // Chat inicia fechado
  const [messages, setMessages] = useState<ChatMessage[]>([
    {
      type: "bot",
      text: "Olá! 👋 Sou o assistente virtual da KRTI Network Solutions. Vou coletar algumas informações para enviar um orçamento personalizado. Vamos começar?",
    },
  ]);
  const [currentStep, setCurrentStep] = useState(0);
  const [formData, setFormData] = useState<FormData>({
    name: "",
    email: "",
    phone: "",
    address: "",
  });
  const [isSubmitting, setIsSubmitting] = useState(false);
  const messagesEndRef = useRef<HTMLDivElement>(null);
  const inputRef = useRef<HTMLInputElement>(null);
  const { toast } = useToast();

  const steps = [
    {
      question: "Qual é o seu nome completo?",
      field: "name" as keyof FormData,
    },
    {
      question: "Qual é o seu e-mail?",
      field: "email" as keyof FormData,
    },
    {
      question: "Qual é o seu telefone?",
      field: "phone" as keyof FormData,
    },
    {
      question: "Qual é o seu endereço completo?",
      field: "address" as keyof FormData,
    },
  ];

  useEffect(() => {
    if (isOpen && inputRef.current) {
      inputRef.current.focus();
    }
  }, [isOpen, currentStep]);

  useEffect(() => {
    messagesEndRef.current?.scrollIntoView({ behavior: "smooth" });
  }, [messages]);

  const handleInputChange = (value: string) => {
    const currentField = steps[currentStep].field;
    setFormData((prev) => ({ ...prev, [currentField]: value }));
  };

  const handleSend = async () => {
    const currentField = steps[currentStep].field;
    const value = formData[currentField].trim();

    if (!value) {
      toast({
        title: "Campo obrigatório",
        description: "Por favor, preencha este campo.",
        variant: "destructive",
      });
      return;
    }

    // Add user message
    setMessages((prev) => [...prev, { type: "user", text: value }]);

    // Move to next step
    if (currentStep < steps.length - 1) {
      setCurrentStep((prev) => prev + 1);
      setTimeout(() => {
        setMessages((prev) => [
          ...prev,
          { type: "bot", text: steps[currentStep + 1].question },
        ]);
      }, 500);
    } else {
      // All data collected, submit
      await submitForm();
    }

    // Clear input
    setFormData((prev) => ({ ...prev, [currentField]: "" }));
  };

  const submitForm = async () => {
    setIsSubmitting(true);
    setMessages((prev) => [
      ...prev,
      {
        type: "bot",
        text: "Obrigado pelas informações! Estou processando e enviando para nossa equipe...",
      },
    ]);

    try {
      const response = await fetch("/api/send-excel", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(formData),
      });

      const result = await response.json();

      if (result.success) {
        setMessages((prev) => [
          ...prev,
          {
            type: "bot",
            text: "✅ Perfeito! Seus dados foram enviados com sucesso para nossa equipe. Entraremos em contato em breve!",
          },
        ]);
        toast({
          title: "Sucesso!",
          description: "Dados enviados com sucesso!",
        });
        
        // Reset form
        setTimeout(() => {
          setIsOpen(false);
          setCurrentStep(0);
          setFormData({
            name: "",
            email: "",
            phone: "",
            address: "",
          });
          setMessages([
            {
              type: "bot",
              text: "Olá! 👋 Sou o assistente virtual da KRTI Network Solutions. Vou coletar algumas informações para enviar um orçamento personalizado. Vamos começar?",
            },
          ]);
        }, 3000);
      } else {
        throw new Error(result.message || "Erro ao enviar dados");
      }
    } catch (error) {
      console.error("Error submitting form:", error);
      setMessages((prev) => [
        ...prev,
        {
          type: "bot",
          text: "❌ Ops! Ocorreu um erro ao enviar seus dados. Por favor, tente novamente ou entre em contato diretamente pelo WhatsApp.",
        },
      ]);
      toast({
        title: "Erro",
        description: "Erro ao enviar dados. Tente novamente.",
        variant: "destructive",
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      handleSend();
    }
  };

  return (
    <>
      {/* Floating Button - Chat Bot (RIGHT SIDE ONLY) */}
      <button
        onClick={(e) => {
          e.preventDefault();
          e.stopPropagation();
          setIsOpen(!isOpen);
        }}
        className="fixed right-6 bottom-6 z-50 group"
        aria-label="Abrir chat bot"
        type="button"
        style={{ right: '24px', bottom: '24px' }}
      >
        <div className="relative">
          {/* Glow effect */}
          <div className=""></div>
          
          {/* Button */}
          <div className="relative bg-gradient-to-br from-primary to-blue-600 hover:from-blue-600 hover:to-blue-700 text-white rounded-full p-4 shadow-2xl transition-all duration-300 group-hover:scale-110 group-hover:shadow-primary/50 tech-border-glow">
            {isOpen ? <X className="w-6 h-6" /> : <Bot className="w-6 h-6" />}
          </div>
        </div>
      </button>

      {/* Chat Window - Sidebar from right */}
      <AnimatePresence>
        {isOpen && (
          <>
            {/* Backdrop */}
            <motion.div
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              exit={{ opacity: 0 }}
              transition={{ duration: 0.3 }}
              className="fixed right-6  bg-black/50 z-50"
              onClick={() => setIsOpen(false)}
            />
            
            {/* Sidebar - Slides in from RIGHT side when isOpen is true */}
            <motion.div
              initial={{ x: "100%" }}
              animate={{ x: 0 }}
              exit={{ x: "100%" }}
              transition={{ type: "spring", damping: 25, stiffness: 200 }}
              className="fixed right-0 top-0 h-full w-full max-w-md z-50 flex flex-col glass-tech shadow-2xl tech-border-glow overflow-hidden"
              onClick={(e) => e.stopPropagation()}
              style={{
                right: '-70%',
                left: 'auto',
              }}
            >
            {/* Header */}
            <div className="bg-gradient-to-r from-primary to-blue-600 text-white p-4 flex items-center justify-between flex-shrink-0">
              <div className="flex items-center gap-2">
                <Bot className="w-5 h-5" />
                <h3 className="font-semibold">Assistente Virtual</h3>
              </div>
              <button
                onClick={() => setIsOpen(false)}
                className="hover:bg-white/20 rounded-full p-1 transition-colors"
              >
                <X className="w-4 h-4" />
              </button>
            </div>

            {/* Messages */}
            <div className="flex-1 overflow-y-auto p-4 space-y-4 bg-background/50">
            {messages.map((message, index) => (
              <div
                key={index}
                className={`flex gap-2 ${
                  message.type === "user" ? "justify-end" : "justify-start"
                }`}
              >
                {message.type === "bot" && (
                  <div className="w-8 h-8 rounded-full bg-primary/20 flex items-center justify-center flex-shrink-0">
                    <Bot className="w-4 h-4 text-primary" />
                  </div>
                )}
                <div
                  className={`max-w-[80%] rounded-lg p-3 ${
                    message.type === "user"
                      ? "bg-primary text-white"
                      : "bg-muted text-foreground"
                  }`}
                >
                  <p className="text-sm whitespace-pre-wrap">{message.text}</p>
                </div>
                {message.type === "user" && (
                  <div className="w-8 h-8 rounded-full bg-primary/20 flex items-center justify-center flex-shrink-0">
                    <User className="w-4 h-4 text-primary" />
                  </div>
                )}
              </div>
            ))}
            {isSubmitting && (
              <div className="flex gap-2 justify-start">
                <div className="w-8 h-8 rounded-full bg-primary/20 flex items-center justify-center flex-shrink-0">
                  <Bot className="w-4 h-4 text-primary animate-pulse" />
                </div>
                <div className="bg-muted text-foreground rounded-lg p-3">
                  <div className="flex gap-1">
                    <span className="w-2 h-2 bg-foreground/50 rounded-full animate-bounce"></span>
                    <span className="w-2 h-2 bg-foreground/50 rounded-full animate-bounce" style={{ animationDelay: "0.2s" }}></span>
                    <span className="w-2 h-2 bg-foreground/50 rounded-full animate-bounce" style={{ animationDelay: "0.4s" }}></span>
                  </div>
                </div>
              </div>
            )}
            <div ref={messagesEndRef} />
          </div>

          {/* Input */}
          {currentStep < steps.length && !isSubmitting && (
            <div className="p-4 border-t border-border bg-background/50">
              <div className="flex gap-2">
                <Input
                  ref={inputRef}
                  type="text"
                  placeholder={steps[currentStep].question}
                  value={formData[steps[currentStep].field]}
                  onChange={(e) => handleInputChange(e.target.value)}
                  onKeyPress={handleKeyPress}
                  className="flex-1"
                />
                <Button
                  onClick={handleSend}
                  className="bg-primary hover:bg-blue-700"
                  disabled={!formData[steps[currentStep].field].trim()}
                >
                  <Send className="w-4 h-4" />
                </Button>
              </div>
            </div>
          )}
            </motion.div>
          </>
        )}
      </AnimatePresence>
    </>
  );
}

